from flask import Flask, g, redirect, render_template, request, url_for
from flask_caching import Cache
import json
import psycopg2
import psycopg2.extras
import time

app = Flask(__name__)
app.secret_key = 'visage-secret-key';

cache = Cache(app, config={"CACHE_TYPE": "RedisCache"})

@app.before_request
def before_request():
    g.start = time.time()

@app.after_request
def after_request(response):
    diff = time.time() - g.start
    if ((response.response) and
         (200 <= response.status_code < 300) and
         (response.content_type.startswith('text/html'))):
        response.set_data(response.get_data().replace(
            b'__EXECUTION_TIME__', bytes(str(diff), 'utf-8')))
    return response

@app.route("/", methods=['GET'])
def homepage():

    # Most recent public company blog post.
    latest_blog_post = cache.get("latest_blog_post") or {}

    # A count of recently active users.
    recent_user_count = cache.get("recent_user_count") or {'count':0}

    #  Four of the latest posts.
    recent_posts = cache.get("recent_posts") or []

    return render_template('home.html',
                           latest_blog_post=latest_blog_post,
                           recent_user_count=recent_user_count,
                           recent_posts=recent_posts)

@app.route("/post_form", methods=['GET'])
def post():
    return render_template("post_form.html")

@app.route("/post_submit", methods=['POST'])
def post_submit():

    with psycopg2.connect("dbname='visage' host='localhost'") as conn:

        with conn.cursor() as cur:

            # Insert new post to database
            cur.execute("""
                INSERT INTO post (post_title, post_text, user_id)
                VALUES (%(post_title)s, %(post_text)s, %(user_id)s)
                RETURNING post_id
            """,
                        {'post_title': request.form['post_title'],
                         'post_text': request.form['post_text'],
                         'user_id': 1})
            new_post_id = cur.fetchone()[0]

        # Update cache to add the new post just created,
        # and hold no more than 4 posts
        recent_posts = cache.get("recent_posts") or []
        recent_posts = recent_posts[-3:]

        new_post = {
            "post_id": new_post_id,
            "post_title": request.form['post_title'],
            "post_text_short": request.form['post_text'][:100]
        }

        recent_posts.append(new_post)
        cache.set("recent_posts", recent_posts)

    return redirect('/')

if __name__ == '__main__':
    app.run()
