from flask import Flask, g, render_template
import psycopg2
import psycopg2.extras
import time

app = Flask(__name__)

from flask_caching import Cache

cache = Cache(app, config={"CACHE_TYPE": "RedisCache"})

@app.before_request
def before_request():
    g.start = time.time()

@app.after_request
def after_request(response):
    diff = time.time() - g.start
    if ((response.response) and
         (200 <= response.status_code < 300) and
         (response.content_type.startswith('text/html'))):
        response.set_data(response.get_data().replace(
            b'__EXECUTION_TIME__', bytes(str(diff), 'utf-8')))
    return response

@app.route("/", methods=['GET'])
def homepage():

    with psycopg2.connect("dbname='visage' host='localhost'") as conn:

        with conn.cursor(cursor_factory=psycopg2.extras.RealDictCursor) as cur:

            # Most recent public company blog post.
            latest_blog_post = cache.get("latest_blog_post")
            if latest_blog_post is None:
                cur.execute("""
                    SELECT post_id, post_title,
                        LEFT(post_text, 100) AS post_text_short
                    FROM post JOIN "user" USING (user_id)
                    WHERE user_type_id = 1
                    ORDER BY updated_at DESC LIMIT 1
                """)
                latest_blog_post = cur.fetchone()
                cache.set("latest_blog_post", latest_blog_post)
                time.sleep(3.33) # simulate long-running query
                app.log.debug("read latest blog post from SQL")
            else:
                app.log.debug("read latest blog post from cache")

            # A count of recently active users.
            recent_user_count = cache.get("recent_user_count")
            if recent_user_count is None:
                cur.execute("""
                    SELECT COUNT(DISTINCT user_id) AS count
                    FROM (
                        SELECT user_id FROM comment
                        WHERE created_at > NOW() - INTERVAL '1' DAY
                        UNION
                        SELECT user_id FROM post
                        WHERE updated_at > NOW() - INTERVAL '1' DAY
                    ) AS u;
                """)
                recent_user_count = cur.fetchone()
                cache.set("recent_user_count", recent_user_count)
                time.sleep(3.33) # simulate long-running query
                app.log.debug("read user count from SQL")
            else:
                app.log.debug("read user count from cache")

            #  Four of the latest posts.
            recent_posts = cache.get("recent_posts")
            if recent_posts is None:
                cur.execute("""
                    SELECT post_id, post_title,
                        LEFT(post_text, 100) AS post_text_short
                    FROM post
                    ORDER BY updated_at DESC LIMIT 4
                """)
                recent_posts = cur.fetchall()
                cache.set("recent_posts", recent_posts)
                time.sleep(3.33) # simulate long-running query
                app.log.debug("read recent posts from SQL")
            else:
                app.log.debug("read recent posts from cache")

    return render_template('home.html',
                           latest_blog_post=latest_blog_post,
                           recent_user_count=recent_user_count,
                           recent_posts=recent_posts)

if __name__ == '__main__':
    app.run()
