from datetime import datetime, timezone
from flask import Flask, redirect, render_template, request
from flask_sqlalchemy import SQLAlchemy
from sqlalchemy import ForeignKey, func
from sqlalchemy.orm import DeclarativeBase, Mapped, mapped_column, relationship
from time import sleep
from typing import List

flask_app = Flask(__name__)
flask_app.config.from_mapping(
    SECRET_KEY="visage_key",
    SQLALCHEMY_DATABASE_URI="postgresql+psycopg2://localhost:5432/visage",
    SQLALCHEMY_ECHO=True,
)

class Base(DeclarativeBase):
  pass

db = SQLAlchemy(model_class=Base)
db.init_app(flask_app)

class Post(db.Model):
    __tablename__ = "post"

    post_id: Mapped[int] = mapped_column(primary_key=True)
    post_title: Mapped[str]
    post_text: Mapped[str]
    created_at: Mapped[datetime] = mapped_column(
            default=lambda: datetime.now(timezone.utc))
    updated_at: Mapped[datetime] = mapped_column(
            default=lambda: datetime.now(timezone.utc),
            onupdate=lambda: datetime.now(timezone.utc))
    user_id: Mapped[int] = mapped_column(ForeignKey("user.user_id"), nullable=False)
    user: Mapped["User"] = relationship("User", back_populates="posts")

    @staticmethod
    def get_recent(n):
        with db.session.begin():
            recent_posts = db.session.execute(
                db.select(
                    Post.post_title,
                    func.left(Post.post_text, 100).label("post_text_short"),
                    User.display_name).
                select_from(Post).
                join(Post.user).
                order_by(Post.updated_at.desc()).
                limit(4)
            )
        return recent_posts

class User(db.Model):
    __tablename__ = "user"

    user_id: Mapped[int] = mapped_column(primary_key=True)
    user_type_id: Mapped[int]
    display_name: Mapped[str]
    real_name: Mapped[str]
    latest_post_id: Mapped[int]
    friend_user_id: Mapped[int] = mapped_column(ForeignKey("user.user_id"))
    posts: Mapped[List["Post"]] = relationship(back_populates="user")

    @staticmethod
    def set_friend(user_id, friend_user_id):
        with db.session.begin():
            db.session.execute(
                db.update(User).
                where(User.user_id == user_id).
                values(friend_user_id = friend_user_id)
            )

        with db.session.begin():
            db.session.execute(
                db.update(User).
                where(User.user_id == friend_user_id).
                values(friend_user_id = user_id)
            )

@flask_app.route("/set_friend", methods=['POST'])
def set_friend_submit():
    User.set_friend(request.form["user_id"], request.form["friend_user_id"])
    return redirect('/users')

@flask_app.route("/users", methods=['GET'])
def users():
    users = db.session.execute(
        db.select(User.display_name, User.user_id, User.friend_user_id).
        order_by(User.display_name)
    ).fetchall()
    return render_template("users.html",
                           users=users)

@flask_app.route("/")
def home():
    recent_posts = Post.get_recent(4)
    return render_template("home.html",
                           recent_posts=recent_posts)

if __name__ == "__main__":
    flask_app.run(port=8080, debug=True)
