from celery import Celery, Task, shared_task 
from flask import Flask, g, redirect, render_template, request
import psycopg2
from psycopg2.extras import RealDictCursor
import time

def celery_init_app(app: Flask) -> Celery:

    class FlaskTask(Task):
        def __call__(self, *args: object, **kwargs: object) -> object:
            with app.app_context():
                return self.run(*args, **kwargs)

    celery_app = Celery(app.name, task_cls=FlaskTask)
    celery_app.config_from_object(app.config["CELERY"])
    celery_app.set_default()
    app.extensions["celery"] = celery_app
    return celery_app

def create_app() -> Flask:
    app = Flask(__name__)
    app.config.from_mapping(
        CELERY=dict(
            broker_url="redis://localhost",
            result_backend="redis://localhost",
            task_ignore_result=True,
        ),
    )
    app.config.from_prefixed_env()
    celery_init_app(app)
    return app

flask_app = create_app()
celery_app = flask_app.extensions["celery"]

@flask_app.before_request
def before_request():
    g.start = time.time()

@flask_app.after_request
def after_request(response):
    diff = time.time() - g.start
    if ((response.response) and
         (200 <= response.status_code < 300) and
         (response.content_type.startswith('text/html'))):
        response.set_data(response.get_data().replace(
            b'__EXECUTION_TIME__', bytes(str(diff), 'utf-8')))
    return response

def process_notifications(post_id):
    flask_app.logger.info(f"Notify for post_id: {post_id}")
    time.sleep(5)

@shared_task()
def post_notify(post_id):

    process_notifications(post_id)

    with psycopg2.connect("dbname='visage' host='localhost'") as conn:
        with conn.cursor() as cur:
            cur.execute("""
                UPDATE post SET notified_at = NOW()
                WHERE post_id = %(post_id)s
            """, {'post_id': post_id})

@flask_app.route("/", methods=['GET'])
def home():

    with psycopg2.connect("dbname='visage' host='localhost'") as conn:
        with conn.cursor(cursor_factory=RealDictCursor) as cur:
            cur.execute("""
                SELECT post_id, post_title,
                    LEFT(post_text, 100) AS post_text_short
                FROM post
                ORDER BY updated_at DESC LIMIT 4
            """)
            recent_posts = cur.fetchall()

    return render_template('home.html',
                           recent_posts=recent_posts)

@flask_app.route("/post_form", methods=['GET'])
def post():
    return render_template("post_form.html")

@flask_app.route("/post_submit", methods=['POST'])
def post_submit():

    with psycopg2.connect("dbname='visage' host='localhost'") as conn:

        with conn.cursor() as cur:

            # Insert new post to database
            cur.execute("""
                INSERT INTO post (post_title, post_text, user_id)
                VALUES (%(post_title)s, %(post_text)s, %(user_id)s)
                RETURNING post_id
            """,
                        {'post_title': request.form['post_title'],
                         'post_text': request.form['post_text'],
                         'user_id': 1})
            new_post_id = cur.fetchone()[0]

    # Send message to a Celery task
    post_notify.delay(new_post_id)
    flask_app.logger.info(f"Enqueued new post_id: {new_post_id}")

    return redirect('/')

if __name__ == "__main__":
    flask_app.run(port=8080, debug=True)
