from datetime import datetime, timezone
from flask import Flask, redirect, render_template, request
from flask_sqlalchemy import SQLAlchemy
from sqlalchemy import ForeignKey, func, update
from sqlalchemy.orm import DeclarativeBase, Mapped, mapped_column, relationship
from typing import List

flask_app = Flask(__name__)
flask_app.config.from_mapping(
    SECRET_KEY="visage_key",
    SQLALCHEMY_DATABASE_URI="postgresql+psycopg2://localhost:5432/visage",
    SQLALCHEMY_ECHO=True,
)

class Base(DeclarativeBase):
  pass

db = SQLAlchemy(model_class=Base)
db.init_app(flask_app)

class Post(db.Model):
    __tablename__ = "post"

    post_id: Mapped[int] = mapped_column(primary_key=True)
    post_title: Mapped[str]
    post_text: Mapped[str]
    created_at: Mapped[datetime] = mapped_column(
            default=lambda: datetime.now(timezone.utc))
    updated_at: Mapped[datetime] = mapped_column(
            default=lambda: datetime.now(timezone.utc),
            onupdate=lambda: datetime.now(timezone.utc))
    user_id: Mapped[int] = mapped_column(ForeignKey("user.user_id"), nullable=False)
    user: Mapped["User"] = relationship("User", back_populates="posts")

    @staticmethod
    def get_recent(n):
        recent_posts = db.session.execute(
            db.select(
                Post.post_title,
                func.left(Post.post_text, 100).label("post_text_short"),
                User.display_name).
            select_from(Post).
            join(Post.user).
            order_by(Post.updated_at.desc()).
            limit(4)
        )
        return recent_posts

    @staticmethod
    def create(user_id, title, text):
        post = Post(post_title=title, post_text=text, user_id=user_id)
        db.session.add(post)
        db.session.flush()

        User.set_latest(user_id, post.post_id)

class User(db.Model):
    __tablename__ = "user"

    user_id: Mapped[int] = mapped_column(primary_key=True)
    user_type_id: Mapped[int]
    display_name: Mapped[str]
    real_name: Mapped[str]
    latest_post_id: Mapped[int]
    posts: Mapped[List["Post"]] = relationship(back_populates="user")

    @staticmethod
    def set_latest(user_id, post_id):
        db.session.execute(
            update(User).
            where(User.user_id == user_id).
            values(latest_post_id = post_id)
        )

@flask_app.route("/")
def home():
    recent_posts = Post.get_recent(4)
    return render_template("home.html",
                           recent_posts=recent_posts)

@flask_app.route("/post_form", methods=['GET'])
def post():
    return render_template("post_form.html")

@flask_app.route("/post_submit", methods=['POST'])
def post_submit():
    with db.session.begin():
        Post.create(1, request.form["post_title"], request.form["post_text"])
    return redirect('/')

if __name__ == "__main__":
    flask_app.run(port=8080, debug=True)
