#---
# Excerpted from "Eloquent Ruby",
# published by The Pragmatic Bookshelf.
# Copyrights apply to this code. It may not be used to create training material,
# courses, books, articles, and the like. Contact us if you are in doubt.
# We make no guarantees that this code is fit for any purpose.
# Visit https://pragprog.com/titles/eruby2 for more book information.
#---
# Simple command line utility that computes statistics
# for Ruby code.
#
# @example
#
#   ruby code_stats.rb activerecord/lib
#
# Note that the numbers generated by this script are very
# rough. In particular it computes the lines/method estimate
# by dividing the number of non-blank/non-comment lines by
# the number of methods. This is almost always be an over
# estimate because it include class and file level declarations
# (e.g. attr_reader :count) in the total number of lines.
#
require "find"

# Call the block for each Ruby source file starting
# with the given path and going recursively down.
def each_ruby_file(starting_path, &block)
  Find.find(starting_path) do |path|
    next unless path.end_with?(".rb")
    block.call(path)
  end
end

# Given the path of a Ruby file, call the block
# for each line of code.
def each_line(starting_path, &block)
  each_ruby_file(starting_path) do |path|
    File.foreach(path) do |line|
      block.call line 
    end
  end
end

def all_comment?(line) = line.match?(/^ *#/)

def blank?(line) = line.match?(/^ #/)

def code?(line)
  return false if blank?(line)
  return false if all_comment?(line)
  true
end

def class?(line) = line.match?(/^ *class [A-Z]/)

def module?(line) = line.match?(/^ *module [A-Z]/)

def method?(line) = line.match?(/^ *def [a-z]/)

def compute_statistics(path)
  lines = 0
  loc = 0
  methods = 0
  classes = 0
  modules = 0

  each_line(ARGV[0]) do |line|
    lines += 1
    loc += 1 if code?(line)
    modules += 1 if module?(line)
    classes += 1 if class?(line)
    methods += 1 if method?(line)
  end

  {lines:, loc:, methods:, classes:, modules:}
end

stats = compute_statistics(ARGV[0])

puts "Total # lines: #{stats[:lines]}"
puts "Actual code lines: #{stats[:loc]}"
puts "Modules: #{stats[:modules]}"
puts "Classes: #{stats[:classes]}"
puts "Methods: #{stats[:methods]}"

lperm = stats[:loc].fdiv(stats[:methods])
lperc = stats[:loc].fdiv(stats[:classes])

puts "Lines / method #{lperm}"
puts "Lines / class #{lperc}"

