from typing import TextIO
from io import StringIO

def read_molecule(reader: TextIO) -> list:
    """Read a single molecule from reader and return it, or return None to
    signal end of file.  The first item in the result is the name of the
    compound; each list contains an atom type and the X, Y, and Z coordinates
    of that atom.

    >>> instring = ('COMPND TEST\\nATOM 1 N 0.1 0.2 0.3\\n' +
    'ATOM 2 N 0.2 0.1 0.0\\nEND\\n')
    >>> infile = StringIO(instring)
    >>> read_molecule(infile)
    ['TEST', ['N', '0.1', '0.2', '0.3'], ['N', '0.2', '0.1', '0.0']]
    """

    # If there isn't another line, we're at the end of the file.
    line = reader.readline()
    if not line:
        return None

    # Name of the molecule: "COMPND   name"
    parts = line.split()
    name = parts[1]

    # Other lines are either "END" or "ATOM num atom_type x y z"
    molecule = [name]

    line = reader.readline()
    while not line.startswith('END'):
        parts = line.split()
        molecule.append(parts[2:])
        line = reader.readline()
    return molecule

def read_all_molecules(reader: TextIO) -> list:
    """Read zero or more molecules from reader, returning a list of the
    molecule information.

    >>> cmpnd1 = ('COMPND T1\\nATOM 1 N 0.1 0.2 0.3\\n' +
    'ATOM 2 N 0.2 0.1 0.0\\nEND\\n')
    >>> cmpnd2 = ('COMPND T2\\nATOM 1 A 0.1 0.2 0.3\\n' +
    'ATOM 2 A 0.2 0.1 0.0\\nEND\\n')
    >>> infile = StringIO(cmpnd1 + cmpnd2)
    >>> result = read_all_molecules(infile)
    >>> result[0]
    ['T1', ['N', '0.1', '0.2', '0.3'], ['N', '0.2', '0.1', '0.0']]
    >>> result[1]
    ['T2', ['A', '0.1', '0.2', '0.3'], ['A', '0.2', '0.1', '0.0']]
    """

    # The list of molecule information.
    result = []

    molecule = read_molecule(reader)
    while molecule:  # None is treated as False in a while loop
        result.append(molecule)
        molecule = read_molecule(reader)
    return result

if __name__ == '__main__':
    with open('multimol.pdb', 'r') as molecule_file:
        molecules = read_all_molecules(molecule_file)
    print(molecules)
