from atom import Atom

class Molecule:
    """A molecule with a name and a list of Atoms. """

    def __init__(self, name: str) -> None:
        """Create a Molecule named name with no Atoms.
        """

        self.name = name
        self.atoms = []

    def add(self, a: Atom) -> None:
        """Add a to my list of Atoms.
        """

        self.atoms.append(a)

    def translate(self, x: float, y: float, z: float) -> None:
        """Move this Molecule, including all Atoms, by (x, y, z).
        """

        for atom in self.atoms:
            atom.translate(x, y, z)

    def __str__(self) -> str:
        """Return a string representation of this Molecule in this format:
            (NAME, (ATOM1, ATOM2, ...))
        """

        res = ''
        for atom in self.atoms:
            res += f'{atom}, '

        # Strip off the last comma.
        res = res[:-2]
        return f'({self.name}, ({res}))'

    def __repr__(self) -> str:
        """Return a string representation of this Molecule in this format:
          Molecule("NAME", (ATOM1, ATOM2, ...))
        """

        res = ''
        for atom in self.atoms:
            res += f'{repr(atom)}, '

        # Strip off the last comma.
        res = res[:-2]
        return f'Molecule("{self.name}", ({res}))'


if __name__ == '__main__':
    ammonia = Molecule("AMMONIA")
    ammonia.add(Atom(1, "N", 0.257, -0.363, 0.0))
    ammonia.add(Atom(2, "H", 0.257, 0.727, 0.0))
    ammonia.add(Atom(3, "H", 0.771, -0.727, 0.890))
    ammonia.add(Atom(4, "H", 0.771, -0.727, -0.890))
    ammonia.translate(0, 0, 0.2)
    assert ammonia.atoms[0].center[0] == 0.257
    assert ammonia.atoms[0].center[1] == -0.363
    assert ammonia.atoms[0].center[2] == 0.2
    print(repr(ammonia))
    print(ammonia)
